package com.apporchid.drivers.repository;

import org.springframework.jdbc.core.ColumnMapRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

import java.sql.ResultSetMetaData;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

@Repository
public class AOPlatformJdbcRepository {

    private static final Logger LOGGER = Logger.getLogger(AOPlatformJdbcRepository.class.getName());

    private final JdbcTemplate jdbcTemplate;

    public AOPlatformJdbcRepository(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    /**
     * Stream rows of the given MSO query. The caller should close the stream.
     */
    public Stream<Map<String, Object>> streamMsoQueryAsMap(String query) {
        LOGGER.log(Level.INFO, "Executing MSO query in streaming mode: {0}", query);
        // ColumnMapRowMapper maps each row into a Map<String, Object> by column name
        return jdbcTemplate.queryForStream(query, new ColumnMapRowMapper());
    }

    /**
     * Stream rows of the given EA query.
     * Alternatively, you can use the same method for MSO or EA queries if they are basically identical.
     */
    public Stream<Map<String, Object>> streamEaQueryAsMap(String query) {
        LOGGER.log(Level.INFO, "Executing EA query in streaming mode: {0}", query);
        return jdbcTemplate.queryForStream(query, new ColumnMapRowMapper());
    }

    /**
     * Streams rows of an MSO query as dynamic RowData objects.
     * Each RowData contains parallel arrays: columnNames and columnValues.
     */
    public Stream<RowData> streamMsoQuery(String query) {
        LOGGER.info("Executing MSO query in streaming mode: " + query);

        return jdbcTemplate.queryForStream(query, (rs, rowNum) -> {
            ResultSetMetaData meta = rs.getMetaData();
            int columnCount = meta.getColumnCount();

            String[] columnNames = new String[columnCount];
            Object[] columnValues = new Object[columnCount];

            for (int i = 0; i < columnCount; i++) {
                columnNames[i] = meta.getColumnName(i + 1);
                columnValues[i] = rs.getObject(i + 1);
            }

            return new RowData(columnNames, columnValues);
        });
    }

    /**
     * Streams rows of an EA query as dynamic RowData objects.
     * Identical approach to MSO, but separated if you need different logic.
     */
    public Stream<RowData> streamEaQuery(String query) {
        LOGGER.info("Executing EA query in streaming mode: " + query);

        return jdbcTemplate.queryForStream(query, (rs, rowNum) -> {
            ResultSetMetaData meta = rs.getMetaData();
            int columnCount = meta.getColumnCount();

            String[] columnNames = new String[columnCount];
            Object[] columnValues = new Object[columnCount];

            for (int i = 0; i < columnCount; i++) {
                columnNames[i] = meta.getColumnName(i + 1);
                columnValues[i] = rs.getObject(i + 1);
            }

            return new RowData(columnNames, columnValues);
        });
    }

    /**
     * A simple record to hold the row data.
     * Each row has parallel arrays of columnNames and columnValues.
     */
    // Use a normal class instead of a record:
    public static class RowData {
        private final String[] columnNames;
        private final Object[] columnValues;

        public RowData(String[] columnNames, Object[] columnValues) {
            this.columnNames = columnNames;
            this.columnValues = columnValues;
        }

        public String[] getColumnNames() {
            return columnNames;
        }

        public Object[] getColumnValues() {
            return columnValues;
        }
    }

}
