package com.apporchid.drivers.service;

import com.apporchid.drivers.repository.AOPlatformJdbcRepository;
import com.apporchid.drivers.repository.AOPlatformJdbcRepository.RowData;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.springframework.stereotype.Service;

import java.io.FileWriter;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

@Service
public class AOPlatformJDBCService {

    private static final Logger LOGGER = Logger.getLogger(AOPlatformJDBCService.class.getName());

    private static final String LOG_FILE    = "results.log";
    private static final String QUERY_MODE  = requiredNonEmptySystemProperty("aoplatform.query.mode");
    private static final String MSO_QUERY   = requiredNonEmptySystemProperty("aoplatform.msoQuery");

    private final AOPlatformJdbcRepository repository;

    public AOPlatformJDBCService(AOPlatformJdbcRepository repository) {
        this.repository = repository;
    }

    public boolean runQuery() {
        return runQuery(MSO_QUERY, LOG_FILE);
    }

    public boolean runQuery(String query, String logFile) {
        if ("MSO".equalsIgnoreCase(QUERY_MODE)) {
            return runMsoQuery(query, logFile);
        } else {
            return runEaQuery(query, logFile);
        }
    }

    private boolean runMsoQuery(String query, String logFile) {
        try (Stream<RowData> rowStream = repository.streamMsoQuery(query)) {
            writeStreamToLogFile(rowStream, logFile ,"MSO");
            return true;
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "MSO Query operation failed.", e);
            return false;
        }
    }

    private boolean runEaQuery(String query, String logFile) {
        try (Stream<RowData> rowStream = repository.streamEaQuery(query)) {
            writeStreamToLogFile(rowStream, logFile,"EA");
            return true;
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "EA Query operation failed.", e);
            return false;
        }
    }

    /**
     * Writes rows in a table format:
     *  1) Print header (column names) once when the first row arrives.
     *  2) For each row, print column values in a fixed-width table style.
     *  3) If no rows appear, prints "No data returned."
     */
    private void writeStreamToLogFile(Stream<RowData> rowStream, String logFile, String subDirectory) throws IOException {
        try (FileWriter fileWriter = new FileWriter("results/"+subDirectory+"/" + logFile)) {
            AtomicBoolean firstRowEncountered = new AtomicBoolean(false);

            // For each row in the stream:
            rowStream.forEach(row -> {
                try {
                    // If this is the first row, print a header
                    if (!firstRowEncountered.getAndSet(true)) {
                        printHeader(fileWriter, row.getColumnNames());
                    }
                    // Now print row values
                    printRow(fileWriter, row.getColumnValues());
                } catch (IOException ex) {
                    throw new RuntimeException("Error writing row to file", ex);
                }
            });

            // If we never saw a row, print a message
            if (!firstRowEncountered.get()) {
                fileWriter.write("No data returned.\n");
            }

            LOGGER.info("\n==============================\n"
                    + "Results written to log file: " + logFile
                    + "\n==============================");
        }
    }

    /**
     * Print the table header (column names) in a fixed-width format, then a separator line.
     */
    private void printHeader(FileWriter fileWriter, String[] columnNames) throws IOException {
        // Build header row
        StringBuilder header = new StringBuilder();
        for (String colName : columnNames) {
            // %-20s => left-justify in a 20-char field
            header.append(String.format("%-20s", colName));
        }
        fileWriter.write(header.toString() + "\n");

        // Build and write a line of dashes (separator)
        fileWriter.write("-".repeat(header.length()) + "\n");
    }

    /**
     * Print the row in the same column width as the header.
     */
    private void printRow(FileWriter fileWriter, Object[] columnValues) throws IOException {
        StringBuilder rowLine = new StringBuilder();
        for (Object value : columnValues) {
            // If value is null, we can just print empty or "null"
            String text = (value == null) ? "null" : value.toString();
            rowLine.append(String.format("%-20s", text));
        }
        fileWriter.write(rowLine.toString() + "\n");
    }

    private static String requiredNonEmptySystemProperty(String propertyName) {
        String value = System.getProperty(propertyName);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(value),
                String.format("System property %s must be non-empty", propertyName));
        return value;
    }
}
