package com.apporchid.drivers.service;

import com.apporchid.drivers.repository.AOPlatformJdbcRepository;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.springframework.stereotype.Service;

import java.io.FileWriter;
import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

@Service
public class AOPlatformJDBCServiceAsMap {

    private static final Logger LOGGER = Logger.getLogger(AOPlatformJDBCServiceAsMap.class.getName());

    private static final String LOG_FILE = "results.log";
    private static final String QUERY_MODE = requiredNonEmptySystemProperty("aoplatform.query.mode");
    private static final String MSO_QUERY = requiredNonEmptySystemProperty("aoplatform.msoQuery");

    private final AOPlatformJdbcRepository repository;

    public AOPlatformJDBCServiceAsMap(AOPlatformJdbcRepository repository) {
        this.repository = repository;
    }

    private static String requiredNonEmptySystemProperty(String propertyName) {
        String value = System.getProperty(propertyName);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(value),
                String.format("System property %s must be non-empty", propertyName));
        return value;
    }

    public boolean runQueryAsMap() {
        return runQueryAsMap(MSO_QUERY, LOG_FILE);
    }

    public boolean runQueryAsMap(String query, String logFile) {
        if ("MSO".equalsIgnoreCase(QUERY_MODE)) {
            return runMsoQueryAsMap(query, logFile);
        } else {
            return runEaQueryAsMap(query, logFile);
        }
    }

    private boolean runMsoQueryAsMap(String query, String logFile) {
        try (Stream<Map<String, Object>> rowStream = repository.streamMsoQueryAsMap(query)) {
            writeStreamToLogFileAsMap(rowStream, logFile + "_MSO");
            return true;
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "MSO Query operation failed.", e);
            return false;
        }
    }

    private boolean runEaQueryAsMap(String query, String logFile) {
        try (Stream<Map<String, Object>> rowStream = repository.streamEaQueryAsMap(query)) {
            writeStreamToLogFileAsMap(rowStream, logFile + "_EA");
            return true;
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "EA Query operation failed.", e);
            return false;
        }
    }

    private void writeStreamToLogFileAsMap(Stream<Map<String, Object>> rowStream, String logFile) throws IOException {
        // Ensure "results" directory exists (optional) or create it
        try (FileWriter fileWriter = new FileWriter("results/" + logFile)) {
            rowStream.forEach(row -> {
                try {
                    fileWriter.write(row.toString());
                    fileWriter.write("\n");
                } catch (IOException e) {
                    throw new RuntimeException("Error writing row to file", e);
                }
            });
            LOGGER.info("\n==============================\nResults written to log file: "
                    + logFile + "\n==============================");
        }
    }

}
