package com.apporchid.drivers;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Builder;
import lombok.NoArgsConstructor;

import java.io.FileWriter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

@NoArgsConstructor
@Builder
public class AOPlatformJDBCExample {

    private static final Logger LOGGER = Logger.getLogger(AOPlatformJDBCExample.class.getName());
    private static final String LOG_FILE = "results.log";
    private static final String JDBC_URL = requiredNonEmptySystemProperty("aoplatform.test.server.url");
    private static final String USER = requiredNonEmptySystemProperty("aoplatform.test.server.user");
    private static final String PASSWORD = requiredNonEmptySystemProperty("aoplatform.test.server.password");
    private static final String SCHEMA = requiredNonEmptySystemProperty("aoplatform.schema.name");
    private static final String QUERY_MODE = requiredNonEmptySystemProperty("aoplatform.query.mode");
    private static final String MSO_QUERY = requiredNonEmptySystemProperty("aoplatform.msoQuery");

    public boolean runQuery() {
        return runQuery(MSO_QUERY, LOG_FILE);
    }

    public boolean runQuery(String msoQuery, String logFile) {
        if(QUERY_MODE.equalsIgnoreCase("MSO")){
            return runMsoQuery(msoQuery, logFile);
        }
        return runEAQuery(msoQuery, logFile);
    }

    private boolean runMsoQuery() {
        return runMsoQuery(MSO_QUERY, LOG_FILE+"_MSO");
    }

    private boolean runEAQuery() {
        return runEAQuery(MSO_QUERY, LOG_FILE+"_EA");
    }

    private boolean runEAQuery(String msoQuery, String logFile) {
        AOPlatformJDBCExample example = new AOPlatformJDBCExample();

        if (!example.loadDriver()) {
            return false;
        }

        try (Connection connection = example.createConnection();
             Statement statement = example.craeteStatement(connection);
             ResultSet resultSet = statement.executeQuery(msoQuery)) {

            example.writeResultSetToLogFile(resultSet,logFile);
        } catch (SQLException | IOException e) {
            LOGGER.log(Level.SEVERE, "Operation failed.", e);
            return false;
        }
        return true;
    }


    private boolean runMsoQuery(String msoQuery, String logFile) {
        AOPlatformJDBCExample example = new AOPlatformJDBCExample();

        if (!example.loadDriver()) {
            return false;
        }

        try (Connection connection = example.createConnection();
             PreparedStatement statement = example.prepareStatement(connection, msoQuery);
             ResultSet resultSet = statement.executeQuery()) {

            example.writeResultSetToLogFile(resultSet,logFile);
        } catch (SQLException | IOException e) {
            LOGGER.log(Level.SEVERE, "Operation failed.", e);
            return false;
        }
        return true;
    }

    /**
     * Loads the Avatica JDBC driver.
     *
     * @return true if the driver is loaded successfully, false otherwise.
     */
    private boolean loadDriver() {
        try {
            Class.forName("org.apache.calcite.avatica.remote.Driver");
            return true;
        } catch (ClassNotFoundException e) {
            LOGGER.log(Level.SEVERE, "Failed to load Avatica JDBC driver.", e);
            return false;
        }
    }

    /**
     * Creates a connection to the database.
     *
     * @return the database connection.
     * @throws SQLException if a database access error occurs.
     */
    private Connection createConnection() throws SQLException {
        Properties connectionProps = new Properties();
        connectionProps.setProperty("SCHEMA", SCHEMA);
        connectionProps.setProperty("QUERY_MODE", QUERY_MODE);
        connectionProps.setProperty("user", USER);
        connectionProps.setProperty("password", PASSWORD);


        return DriverManager.getConnection(JDBC_URL, connectionProps);
    }

    /**
     * Prepares a statement for the given query.
     *
     * @param connection the database connection.
     * @param query      the SQL query to prepare.
     * @return the prepared statement.
     * @throws SQLException if a database access error occurs.
     */
    private PreparedStatement prepareStatement(Connection connection, String query) throws SQLException {
        return connection.prepareStatement(query);
    }

    private Statement craeteStatement(Connection connection) throws SQLException {
        return connection.createStatement();
    }

    /**
     * Writes the result set to a log file.
     *
     * @param resultSet the result set to write.
     * @throws SQLException if a database access error occurs.
     * @throws IOException  if a file writing error occurs.
     */
    private void writeResultSetToLogFile(ResultSet resultSet, String logFile) throws SQLException, IOException {
        try (FileWriter fileWriter = new FileWriter("results\\"+logFile)) {
            int columnCount = resultSet.getMetaData().getColumnCount();

            // Build table header
            StringBuilder header = new StringBuilder();
            for (int i = 1; i <= columnCount; i++) {
                header.append(String.format("%-20s", resultSet.getMetaData().getColumnName(i)));
            }
            fileWriter.write(header.toString() + "\n");
            fileWriter.write("-".repeat(header.length()) + "\n");

            // Build and write each row
            while (resultSet.next()) {
                StringBuilder row = new StringBuilder();
                for (int i = 1; i <= columnCount; i++) {
                    row.append(String.format("%-20s", resultSet.getString(i)));
                }
                fileWriter.write(row.toString() + "\n");
            }

            LOGGER.info("\n==============================\nResults written to log file: " + logFile + "\n==============================");
        }
    }

    private static String requiredNonEmptySystemProperty(String propertyName) {
        String value = System.getProperty(propertyName);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(value), String.format("System property %s must be non-empty", propertyName));
        return value;
    }

}
